/**
 * Bricks Builder — Decimal Counter Element
 *
 * Handles the count-up animation for the custom Decimal Counter element.
 * Uses requestAnimationFrame for smooth animation with ease-out easing.
 *
 * Preserves all native counter features:
 *   - IntersectionObserver trigger (count on scroll into view)
 *   - Popup support
 *   - Duration setting
 *   - Prefix and suffix
 *   - Thousands separator
 *   - Custom decimal separator
 *
 * Bricks version: 2.2+
 *
 * @see https://forum.bricksbuilder.io/t/support-decimal-values-in-counter-element/38059
 */

// Use var so the global function bricksDecimalCounter can reference it even if
// init is deferred (const would cause a TDZ error if BricksFunction isn't ready yet).
var bricksDecimalCounterFn = null;

function bricksDecimalCounterInit() {
  // Already initialized
  if (bricksDecimalCounterFn) return;

  // BricksFunction not available yet — will retry when bricksDecimalCounter() is called
  if (typeof BricksFunction === 'undefined') return;

  bricksDecimalCounterFn = new BricksFunction({
    parentNode: document,
    selector: '.brxe-decimal-counter',

    // Re-initialize on dynamic content events (AJAX pagination, popups, etc.)
    subscribeEvents: [
      'bricks/popup/open',
      'bricks/ajax/pagination/completed',
      'bricks/ajax/load_page/completed',
      'bricks/ajax/query_result/displayed'
    ],

    // Force reinit if counter is inside a popup
    forceReinit: function (element) {
      return element.closest('.brx-popup');
    },

    eachElement: function (element) {
      var settingsRaw = element.dataset.decimalCounterOptions;

      var settings;
      try {
        settings = JSON.parse(settingsRaw);
      } catch (e) {
        return false;
      }

      var countNode = element.querySelector('.count');
      if (!countNode) return false;

      // --- Parse settings ---
      var countFrom     = typeof settings.countFrom === 'number' ? settings.countFrom : 0;
      var countTo       = typeof settings.countTo === 'number' ? settings.countTo : 100;
      var durationInMs  = typeof settings.duration === 'number' ? settings.duration : 1000;
      var decimalPlaces = typeof settings.decimalPlaces === 'number' ? settings.decimalPlaces : 2;
      var thousands     = settings.thousands || false;
      var separator     = settings.separator || ',';
      var decimalSep    = settings.decimalSep || '.';

      // Min duration: 500ms (matches native behavior)
      if (durationInMs < 500) durationInMs = 500;

      // --- Format a number with decimal places, thousands separator, and custom decimal separator ---
      function formatNumber(value) {
        var fixed = value.toFixed(decimalPlaces);
        var parts = fixed.split('.');
        var intPart = parts[0];
        var decPart = parts.length > 1 ? parts[1] : '';

        // Apply thousands separator to integer part
        if (thousands) {
          var intNum = parseInt(intPart, 10);
          var negative = intNum < 0;
          if (negative) intNum = Math.abs(intNum);

          // Use en-US locale to get comma separators, then replace with user-defined separator
          var formatted = intNum.toLocaleString('en-US');

          if (separator && separator !== ',') {
            formatted = formatted.replace(/,/g, separator);
          }

          intPart = (negative ? '-' : '') + formatted;
        }

        // Combine with decimal part
        if (decimalPlaces > 0) {
          return intPart + decimalSep + decPart;
        }

        return intPart;
      }

      // --- Animation using requestAnimationFrame with ease-out easing ---
      function runCounter() {
        // Reset to start value
        countNode.innerText = formatNumber(countFrom);

        var startTime = null;
        var diff = countTo - countFrom;

        function animate(timestamp) {
          if (!startTime) startTime = timestamp;

          var elapsed = timestamp - startTime;
          var progress = Math.min(elapsed / durationInMs, 1);

          // Ease-out quadratic: fast start, smooth deceleration
          var eased = 1 - (1 - progress) * (1 - progress);

          var currentValue = countFrom + diff * eased;
          countNode.innerText = formatNumber(currentValue);

          if (progress < 1) {
            requestAnimationFrame(animate);
          } else {
            // Ensure final value is exact
            countNode.innerText = formatNumber(countTo);
          }
        }

        requestAnimationFrame(animate);
      }

      // --- Trigger animation ---
      // In the builder, run immediately (no scroll trigger needed)
      var isBuilder = document.body && document.body.classList.contains('bricks-is-builder');

      if (isBuilder) {
        runCounter();
        return;
      }

      var popup = countNode.closest('.brx-popup');

      if (popup) {
        // Inside a popup — run immediately if visible
        if (!popup.classList.contains('hide')) {
          runCounter();
        }
      } else {
        // Use Bricks' IntersectionObserver wrapper to trigger on scroll into view
        new BricksIntersect({
          element: element,
          callback: runCounter
        });
      }
    },

    listenerHandler: function (event) {
      if (event && event.type) {
        switch (event.type) {
          case 'bricks/popup/open':
            var popupSettings = {
              parentNode: event.details && event.details.popupElement
                ? event.details.popupElement
                : document
            };
            bricksDecimalCounterFn.run(popupSettings);
            break;
          default:
            bricksDecimalCounterFn.run();
            break;
        }
      }
    }
  });
}

// Global function called by Bricks via runElementScripts (from the $scripts property).
// Also called manually on the frontend since our script isn't in the core init block.
function bricksDecimalCounter() {
  // Lazy init: ensures BricksFunction is available by the time Bricks calls us
  bricksDecimalCounterInit();

  if (bricksDecimalCounterFn) {
    bricksDecimalCounterFn.run();
  }
}

// Try to init immediately (works when bricks-scripts has already loaded)
bricksDecimalCounterInit();

// Auto-run on the frontend: our script isn't called from Bricks' main init block
// in frontend.js, so we need to trigger it ourselves.
if (bricksDecimalCounterFn && document.body && !document.body.classList.contains('bricks-is-builder')) {
  bricksDecimalCounterFn.addEventListeners();
  bricksDecimalCounter();
}
