<?php
/**
 * Decimal Counter Element for Bricks Builder
 *
 * Extends the native counter element with decimal value support.
 * The native counter only supports integers — this element adds:
 *   - Decimal "Count from" and "Count to" values (e.g. 0.11, 4.7, 99.99)
 *   - Configurable decimal places
 *   - Smooth requestAnimationFrame animation with ease-out easing
 *
 * All native counter features are preserved: prefix, suffix, typography,
 * thousands separator, duration, IntersectionObserver trigger, popup support.
 *
 * @see https://forum.bricksbuilder.io/t/support-decimal-values-in-counter-element/38059
 */

use Bricks\Element;

if ( ! defined( 'ABSPATH' ) ) exit;

class Bytflow_Decimal_Counter extends Element {
	public $category = 'bytflow';
	public $name     = 'decimal-counter';
	public $icon     = 'ti-dashboard';
	public $scripts  = [ 'bricksDecimalCounter' ];

	public function get_label() {
		return esc_html__( 'Decimal Counter', 'bytflow-bricks' );
	}

	public function get_keywords(): array
    {
		return [ 'counter', 'decimal', 'number', 'count', 'animate', 'bytflow' ];
	}

	public function enqueue_scripts() {
		wp_enqueue_script(
			'bytflow-decimal-counter',
			plugin_dir_url( __FILE__ ) . 'script.js',
			[ 'bricks-scripts' ],
			BYTFLOW_BRICKS_VERSION,
			true
		);
	}

	public function set_controls() {
		$this->controls['countFrom'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Count from', 'bytflow-bricks' ),
			'type'        => 'text',
			'inline'      => true,
			'placeholder' => '0',
			'description' => esc_html__( 'Supports decimals (e.g. 0.5)', 'bytflow-bricks' ),
		];

		$this->controls['countTo'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Count to', 'bytflow-bricks' ),
			'type'        => 'text',
			'inline'      => true,
			'default'     => '99.99',
			'description' => esc_html__( 'Supports decimals (e.g. 4.7, 99.99)', 'bytflow-bricks' ),
		];

		$this->controls['decimalPlaces'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Decimal places', 'bytflow-bricks' ),
			'type'        => 'number',
			'min'         => 0,
			'max'         => 10,
			'default'     => 2,
			'description' => esc_html__( 'Number of digits after the decimal point. Set to 0 for integers.', 'bytflow-bricks' ),
		];

		$this->controls['duration'] = [
			'tab'            => 'content',
			'label'          => esc_html__( 'Animation in ms', 'bytflow-bricks' ),
			'type'           => 'number',
			'hasDynamicData' => true,
			'placeholder'    => 1000,
		];

		$this->controls['countTypography'] = [
			'tab'   => 'content',
			'label' => esc_html__( 'Typography', 'bytflow-bricks' ),
			'type'  => 'typography',
			'css'   => [
				[
					'property' => 'font',
					'selector' => '.count',
				],
			],
		];

		// Prefix
		$this->controls['prefixSep'] = [
			'tab'   => 'content',
			'label' => esc_html__( 'Prefix', 'bytflow-bricks' ),
			'type'  => 'separator',
		];

		$this->controls['prefix'] = [
			'tab'    => 'content',
			'label'  => esc_html__( 'Prefix', 'bytflow-bricks' ),
			'type'   => 'text',
			'inline' => true,
		];

		$this->controls['prefixTypography'] = [
			'tab'   => 'content',
			'label' => esc_html__( 'Typography', 'bytflow-bricks' ),
			'type'  => 'typography',
			'css'   => [
				[
					'property' => 'font',
					'selector' => '.prefix',
				],
			],
		];

		// Suffix
		$this->controls['suffixSep'] = [
			'tab'   => 'content',
			'label' => esc_html__( 'Suffix', 'bytflow-bricks' ),
			'type'  => 'separator',
		];

		$this->controls['suffix'] = [
			'tab'    => 'content',
			'label'  => esc_html__( 'Suffix', 'bytflow-bricks' ),
			'type'   => 'text',
			'inline' => true,
		];

		$this->controls['suffixTypography'] = [
			'tab'   => 'content',
			'label' => esc_html__( 'Typography', 'bytflow-bricks' ),
			'type'  => 'typography',
			'css'   => [
				[
					'property' => 'font',
					'selector' => '.suffix',
				],
			],
		];

		// A Thousand separator
		$this->controls['thousandSep'] = [
			'tab'   => 'content',
			'label' => esc_html__( 'Separator', 'bytflow-bricks' ),
			'type'  => 'separator',
		];

		$this->controls['thousandSeparator'] = [
			'tab'   => 'content',
			'label' => esc_html__( 'Thousand separator', 'bytflow-bricks' ),
			'type'  => 'checkbox',
		];

		$this->controls['separatorText'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Separator', 'bytflow-bricks' ),
			'type'        => 'text',
			'inline'      => true,
			'placeholder' => ',',
			'required'    => [ 'thousandSeparator', '=', true ],
		];

		// Decimal separator
		$this->controls['decimalSeparator'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Decimal separator', 'bytflow-bricks' ),
			'type'        => 'text',
			'inline'      => true,
			'placeholder' => '.',
			'description' => esc_html__( 'Character for decimal point (default: dot)', 'bytflow-bricks' ),
		];
	}

	public function render() {
		$settings        = $this->settings;
		$count_from      = ! empty( $settings['countFrom'] ) ? floatval( $this->render_dynamic_data( $settings['countFrom'] ) ) : 0;
		$count_to        = ! empty( $settings['countTo'] ) ? floatval( $this->render_dynamic_data( $settings['countTo'] ) ) : 99.99;
		$prefix          = ! empty( $settings['prefix'] ) ? $this->render_dynamic_data( $settings['prefix'] ) : false;
		$suffix          = ! empty( $settings['suffix'] ) ? $this->render_dynamic_data( $settings['suffix'] ) : false;
		$separator_text  = ! empty( $settings['separatorText'] ) ? sanitize_text_field( $this->render_dynamic_data( $settings['separatorText'] ) ) : '';
		$decimal_sep     = ! empty( $settings['decimalSeparator'] ) ? sanitize_text_field( $settings['decimalSeparator'] ) : '.';
		$duration        = ! empty( $settings['duration'] ) ? intval( $this->render_dynamic_data( $settings['duration'] ) ) : 1000;
		$thousands       = ! empty( $settings['thousandSeparator'] ) ? $settings['thousandSeparator'] : '';
		$decimal_places  = isset( $settings['decimalPlaces'] ) ? intval( $settings['decimalPlaces'] ) : 2;

		// Pass settings as JSON data attribute for the JS animation
		$this->set_attribute(
			'_root',
			'data-decimal-counter-options',
			wp_json_encode(
				[
					'countFrom'      => $count_from,
					'countTo'        => $count_to,
					'duration'       => $duration,
					'decimalPlaces'  => $decimal_places,
					'thousands'      => $thousands,
					'separator'      => $separator_text,
					'decimalSep'     => $decimal_sep,
				]
			)
		);

		echo "<div {$this->render_attributes( '_root' )}>";

		if ( $prefix ) {
			echo '<span class="prefix">' . esc_html( $prefix ) . '</span>';
		}

		// Show the starting value formatted with correct decimal places
		echo '<span class="count">' . esc_html( number_format( $count_from, $decimal_places, $decimal_sep, '' ) ) . '</span>';

		if ( $suffix ) {
			echo '<span class="suffix">' . esc_html( $suffix ) . '</span>';
		}

		echo '</div>';
	}
}
